<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Includes\Utils\Module;

use Includes\Utils\Converter;
use Includes\Utils\FileManager;
use Symfony\Component\Yaml\Parser;
use XCart\Domain\ModuleManagerDomain;

class Registry
{
    /**
     * @var array
     */
    private $runtimeCache;

    /**
     * @var Parser
     */
    private $parser;

    /**
     * @var ModuleManagerDomain
     */
    private $moduleManagerDomain;

    public function __construct()
    {
        $this->parser = new Parser();

        $this->moduleManagerDomain = \XCart\Container::getContainer()->get('XCart\Domain\ModuleManagerDomain');
    }

    /**
     * @return array
     */
    public function getModules()
    {
        return $this->executeCachedRuntime(function () {
            $modules = $this->moduleManagerDomain->getAllModules();

            ksort($modules, SORT_NATURAL);

            return $modules;
        });
    }

    /**
     * @return array
     */
    public function getSkinModules()
    {
        return $this->executeCachedRuntime(function () {
            return array_filter($this->getModules(), static function ($module) {
                return $module['type'] === 'skin';
            });
        });
    }

    /**
     * @return array
     */
    public function getEnabledPaymentModuleIds()
    {
        return $this->executeCachedRuntime(function () {
            $result = [];

            foreach ($this->getModules() as $moduleId => $module) {
                if (
                    $module['type'] === 'payment'
                    && $module['isEnabled']
                ) {
                    $result[] = $moduleId;
                }
            }

            return $result;
        });
    }

    /**
     * @return array
     */
    public function getEnabledShippingModuleIds()
    {
        return $this->executeCachedRuntime(function () {
            $result = [];

            foreach ($this->getModules() as $moduleId => $module) {
                if (
                    $module['type'] === 'shipping'
                    && $module['isEnabled']
                ) {
                    $result[] = $moduleId;
                }
            }

            return $result;
        });
    }

    public function getEnabledModuleIds(bool $skipMainClassCheck = false): array
    {
        return $this->executeCachedRuntime(function () use ($skipMainClassCheck) {
            $enabledModuleIds = $this->moduleManagerDomain->getEnabledModuleIds();

            if ($skipMainClassCheck) {
                $result = $enabledModuleIds;
            } else {
                $result = array_filter($enabledModuleIds, static function ($moduleId) {
                    return class_exists(Module::getMainClassName($moduleId))
                        || FileManager::isFileReadable(Module::getMainDataFilePath($moduleId));
                });
            }

            sort($result, SORT_NATURAL);

            return $result;
        }, ['skipMainClassCheck' => $skipMainClassCheck]);
    }

    /**
     * @param string $author
     * @param string $name
     *
     * @return array|null (@see config/dynamic/xcart_modules.yaml)
     * @deprecated since CDev-Core 5.5.0, use \XCart\Domain\ModuleManagerDomain::getModule() instead
     */
    public function getModule($author, $name = null)
    {
        trigger_deprecation('CDev-Core', '5.5.0', '"%s" is deprecated, use \XCart\Domain\ModuleManagerDomain::getModule() instead.', __METHOD__);

        return $this->moduleManagerDomain->getModule(
            Module::buildId($author, $name)
        );
    }

    /**
     * @param string      $author
     * @param string|null $name
     *
     * @return array
     *
     * @deprecated since CDev-Core 5.5.1, use dependsOn field of array returned by \XCart\Domain\ModuleManagerDomain::getModule()
     */
    public function getDependencies($author, $name = null)
    {
        trigger_deprecation('CDev-Core', '5.5.1', '"%s" is deprecated, use dependsOn field of array returned by \XCart\Domain\ModuleManagerDomain::getModule().', __METHOD__);

        $module = $this->moduleManagerDomain->getModule(
            Module::buildId($author, $name)
        );

        return $module['dependsOn'] ?? [];
    }

    /**
     * @param string      $author
     * @param string|null $name
     *
     * @return bool
     * @deprecated since CDev-Core 5.5.0, use \XCart\Domain\ModuleManagerDomain::isEnabled() instead
     */
    public function isModuleEnabled(string $author, ?string $name = null): bool
    {
        trigger_deprecation('CDev-Core', '5.5.0', '"%s" is deprecated, use \XCart\Domain\ModuleManagerDomain::isEnabled() instead.', __METHOD__);

        [$author, $name] = Module::explodeModuleId($author, $name);

        return $this->moduleManagerDomain->isEnabled("{$author}-{$name}");
    }

    /**
     * @param string $path
     * @param array  $params
     *
     * @return string
     */
    public function getServiceURL($path, array $params = [])
    {
        return \XLite::getInstance()->getServiceURL(
            '#/'
            . $path
            . ($params ? ('?' . http_build_query($params)) : '')
        );
    }

    /**
     * @param string      $author
     * @param string|null $name
     *
     * @return string
     */
    public function getModuleServiceURL($author, $name = null)
    {
        $moduleId = Module::buildId($author, $name);

        $module = $this->moduleManagerDomain->getModule($moduleId);

        if ($module) {
            return \XLite::getInstance()->getServiceURL('#/installed-addons', null, ['moduleId' => $module['id']]);
        }

        return \XLite::getInstance()->getAppStoreUrl() . "xsku-{$moduleId}";
    }

    /**
     * @param string      $author
     * @param string|null $name
     *
     * @return string
     */
    public function getModuleSettingsUrl($author, $name = null)
    {
        $moduleId = Module::buildId($author, $name);

        $module = $this->moduleManagerDomain->getModule($moduleId);

        return $module
            ? Converter::buildURL('module', '', ['moduleId' => $module['id']])
            : null;
    }

    /**
     * @param string      $author
     * @param string|null $name
     *
     * @return string[]
     */
    public function getYamlFiles($author, $name = null)
    {
        $sourcePath = Module::getSourcePath($author, $name);

        $result = [
            $sourcePath . 'install.yaml',
        ];

        foreach (glob($sourcePath . 'install_*.yaml') ?: [] as $translationFile) {
            $result[] = $translationFile;
        }

        return $result;
    }

    /** Port of runtime cache trait */

    /**
     * @param callable $callback
     * @param null     $cacheKeyParts
     * @param bool     $force
     *
     * @return mixed
     */
    protected function executeCachedRuntime(callable $callback, $cacheKeyParts = null, $force = false)
    {
        if ($cacheKeyParts === null) {
            $cacheKeyParts = debug_backtrace(\DEBUG_BACKTRACE_IGNORE_ARGS, 2)[1]['function'];
        }

        $cacheKey = $this->getRuntimeCacheKey([$cacheKeyParts]);

        if (!isset($this->runtimeCache[$cacheKey]) || $force) {
            $this->runtimeCache[$cacheKey] = $callback();
        }

        return $this->runtimeCache[$cacheKey];
    }

    /**
     * Calculate key for cache storage
     *
     * @param mixed $cacheKeyParts
     *
     * @return string
     */
    protected function getRuntimeCacheKey($cacheKeyParts)
    {
        return is_scalar($cacheKeyParts) ? (string) $cacheKeyParts : md5(serialize($cacheKeyParts));
    }
}
